<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor image gallery widget.
 *
 * Elementor widget that displays a set of images in an aligned grid.
 *
 * @since 1.0.0
 */
class OSF_Elementor_Image_Gallery extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve image gallery widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'opal-image-gallery';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve image gallery widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Opal Image Gallery', 'worldlife-core' );
	}

	public function get_script_depends() {
		return [
			'modernizr',
			'hoverdir',
			'imagesloaded',
			'masonry',
			'anime',
		];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image gallery widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-gallery-grid';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since  2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'image', 'photo', 'visual', 'gallery' ];
	}

	/**
	 * Register image gallery widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'section_gallery',
			[
				'label' => __( 'Image Gallery', 'worldlife-core' ),
			]
		);

		$this->add_control(
			'wp_gallery',
			[
				'label'      => __( 'Add Images', 'worldlife-core' ),
				'type'       => Controls_Manager::GALLERY,
				'show_label' => false,
				'dynamic'    => [
					'active' => true,
				],
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'thumbnail',
				// Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
				'exclude'   => [ 'custom' ],
				'separator' => 'none',
			]
		);


		$this->add_responsive_control(
			'columns',
			[
				'label'   => __( 'Columns', 'worldlife-core' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 3,
				'options' => [ 1 => 1, 2 => 2, 3 => 3, 4 => 4, 6 => 6 ],
			]
		);

		$this->add_responsive_control(
			'product_gutter',
			[
				'label'      => __( 'Gutter', 'worldlife-core' ),
				'type'       => Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 60,
					],
				],
				'size_units' => [ 'px' ],
				'selectors'  => [
					'{{WRAPPER}} .column-item' => 'padding-left: calc({{SIZE}}{{UNIT}} / 2); padding-right: calc({{SIZE}}{{UNIT}} / 2); padding-bottom: calc({{SIZE}}{{UNIT}})',
					'{{WRAPPER}} .row'         => 'margin-left: calc({{SIZE}}{{UNIT}} / -2); margin-right: calc({{SIZE}}{{UNIT}} / -2);',
				],
			]
		);

		$this->add_control(
			'gallery_animation',
			[
				'label'              => __( 'Entrance Animation Item', 'worldlife-core' ),
				'type'               => Controls_Manager::SELECT,
				'default'            => 'Shu',
				'options'            => [
					'Ra'  => __( 'Ra', 'worldlife-core' ),
					'Shu' => __( 'Shu', 'worldlife-core' ),
					'Nut' => __( 'Nut', 'worldlife-core' )
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'view',
			[
				'label'   => __( 'View', 'worldlife-core' ),
				'type'    => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_gallery_images',
			[
				'label' => __( 'Images', 'worldlife-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'image_spacing',
			[
				'label'        => __( 'Spacing', 'worldlife-core' ),
				'type'         => Controls_Manager::SELECT,
				'options'      => [
					''       => __( 'Default', 'worldlife-core' ),
					'custom' => __( 'Custom', 'worldlife-core' ),
				],
				'prefix_class' => 'gallery-spacing-',
				'default'      => '',
			]
		);

		$columns_margin  = is_rtl() ? '0 0 -{{SIZE}}{{UNIT}} -{{SIZE}}{{UNIT}};' : '0 -{{SIZE}}{{UNIT}} -{{SIZE}}{{UNIT}} 0;';
		$columns_padding = is_rtl() ? '0 0 {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}};' : '0 {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}} 0;';

		$this->add_control(
			'image_spacing_custom',
			[
				'label'      => __( 'Image Spacing', 'worldlife-core' ),
				'type'       => Controls_Manager::SLIDER,
				'show_label' => false,
				'range'      => [
					'px' => [
						'max' => 100,
					],
				],
				'default'    => [
					'size' => 15,
				],
				'selectors'  => [
					'{{WRAPPER}} .gallery-item' => 'padding:' . $columns_padding,
					'{{WRAPPER}} .gallery'      => 'margin: ' . $columns_margin,
				],
				'condition'  => [
					'image_spacing' => 'custom',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'      => 'image_border',
				'selector'  => '{{WRAPPER}} .gallery-item img',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => __( 'Border Radius', 'worldlife-core' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .gallery-item img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_caption',
			[
				'label' => __( 'Caption', 'worldlife-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'gallery_display_caption',
			[
				'label'     => __( 'Display', 'worldlife-core' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '',
				'options'   => [
					''     => __( 'Show', 'worldlife-core' ),
					'none' => __( 'Hide', 'worldlife-core' ),
				],
				'selectors' => [
					'{{WRAPPER}} .gallery-item .gallery-caption' => 'display: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'align',
			[
				'label'     => __( 'Alignment', 'worldlife-core' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => [
					'left'    => [
						'title' => __( 'Left', 'worldlife-core' ),
						'icon'  => 'fa fa-align-left',
					],
					'center'  => [
						'title' => __( 'Center', 'worldlife-core' ),
						'icon'  => 'fa fa-align-center',
					],
					'right'   => [
						'title' => __( 'Right', 'worldlife-core' ),
						'icon'  => 'fa fa-align-right',
					],
					'justify' => [
						'title' => __( 'Justified', 'worldlife-core' ),
						'icon'  => 'fa fa-align-justify',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .gallery-item .gallery-caption' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					'gallery_display_caption' => '',
				],
			]
		);

		$this->add_control(
			'text_color',
			[
				'label'     => __( 'Text Color', 'worldlife-core' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .gallery-item .gallery-caption' => 'color: {{VALUE}};',
				],
				'condition' => [
					'gallery_display_caption' => '',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'      => 'typography',
				'scheme'    => Scheme_Typography::TYPOGRAPHY_4,
				'selector'  => '{{WRAPPER}} .gallery-item .gallery-caption',
				'condition' => [
					'gallery_display_caption' => '',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render image gallery widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		if ( ! $settings['wp_gallery'] ) {
			return;
		}

		$this->add_render_attribute( 'row', 'class', 'row grid' );

		if ( ! empty( $settings['columns'] ) ) {
			$this->add_render_attribute( 'row', 'data-elementor-columns', $settings['columns'] );
		}

		if ( ! empty( $settings['columns_tablet'] ) ) {
			$this->add_render_attribute( 'row', 'data-elementor-columns-tablet', $settings['columns_tablet'] );
		}
		if ( ! empty( $settings['columns_mobile'] ) ) {
			$this->add_render_attribute( 'row', 'data-elementor-columns-mobile', $settings['columns_mobile'] );
		}


		?>
        <div class="elementor-image-gallery">
            <div <?php echo $this->get_render_attribute_string( 'row' ) ?>>
				<?php
				$this->add_render_attribute( 'link', [
					'data-elementor-lightbox-slideshow' => $this->get_id(),
				] );

				if ( ! empty( $settings['open_lightbox'] ) ) {
					$this->add_render_attribute( 'link', [
						'data-elementor-open-lightbox' => $settings['open_lightbox'],
					] );
				}
				if ( Plugin::$instance->editor->is_edit_mode() ) {
					$this->add_render_attribute( 'link', [
						'class' => 'elementor-clickable',
					] );
				}

				foreach ( $settings['wp_gallery'] as $index => $attachment ) {
					$image_url = Group_Control_Image_Size::get_attachment_image_src( $attachment['id'], 'thumbnail', $settings );
					?>
                    <div class="column-item grid__item">
                        <a class="grid__link" <?php echo $this->get_render_attribute_string( 'link' ); ?>
                           href="<?php echo esc_attr( $image_url ); ?>">
                            <img class="" src="<?php echo esc_attr( $image_url ); ?>"
                                 alt="<?php echo esc_attr( Control_Media::get_image_alt( $attachment ) ); ?>"/>
                            <div class="gallery-item-overlay">
                                <i class="fa fa-image"></i>
                                <span><?php echo esc_html__( 'View Gallery', 'worldlife-core' ); ?></span>
                            </div>
                        </a>
                    </div>
				<?php } ?>
            </div>
        </div>
		<?php
	}
}

$widgets_manager->register_widget_type( new OSF_Elementor_Image_Gallery() );
